package com.rockysaas.api.demo.control;

import java.security.Key;
import java.security.PublicKey;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;

import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import com.rockysaas.api.demo.model.OpenProviderReceiveRequestDto;
import com.rockysaas.api.demo.model.OpenResponseDto;
import com.rockysaas.api.demo.util.MapUtil;
import com.rockysaas.api.demo.util.RSAUtil;

import io.swagger.annotations.Api;
import lombok.extern.slf4j.Slf4j;

@RestController
@Slf4j
@RequestMapping(value = "/api", produces = MediaType.APPLICATION_JSON_VALUE)
@Api(value = "Web - VerifyTestController", produces = MediaType.APPLICATION_JSON_VALUE)
public class VerifyTestController {
	// 不参与签名的字段
	private static final Set<String> SIGN_EXCLUDE_SET = new HashSet<>();

	@Value(value = "${rockysaas.open.saas.public-key}")
	private String publicKey;

	static {
		SIGN_EXCLUDE_SET.add("sign");
	}

	@PostMapping(value = "/")
	public OpenResponseDto api(@RequestBody @Valid OpenProviderReceiveRequestDto requestProviderDto,
			HttpServletRequest request, HttpServletResponse response) throws Exception {
		OpenResponseDto responseDto = new OpenResponseDto();

		// 检查appId
		String appId = requestProviderDto.getAppId();

		try {
			// 请求校验
			verifyApiRequest(requestProviderDto);

			return responseDto;

		} catch (Exception e) {
			log.error(e.getMessage(), e);
			responseDto.createError(501, e.getMessage());
			return responseDto;
		}

	}

	public void verifyApiRequest(OpenProviderReceiveRequestDto requestProviderDto) throws Exception {
		// 检查appId

		Map<String, Object> map = MapUtil.getMapFromObject(requestProviderDto);
		boolean verifyResult = verify(map, publicKey);

		if (!verifyResult) {
			log.error("验签失败 ");
			throw new Exception("验签失败 ");
		}
	}

	/**
	 * 
	 * @Description 验签
	 * @author <a href="hugaoxiang8619@adpanshi.com">hugaoxiang</a>
	 * @param parameters
	 * @param publicKeyStr
	 * @return
	 * @throws Exception
	 */
	private boolean verify(Map<String, Object> parameters, String publicKeyStr) throws Exception {

		String sign = parameters.get("sign").toString();

		SortedMap<String, Object> sortedParameters = new TreeMap<>(parameters);
		StringBuffer sb = new StringBuffer();
		for (Map.Entry<String, Object> entry : sortedParameters.entrySet()) {
			String k = entry.getKey();
			if (!SIGN_EXCLUDE_SET.contains(k)) {
				String v = entry.getValue().toString();
				sb.append("&" + k + "=" + v);
			}

		}

		if (sb.length() > 0) {
			String str = sb.deleteCharAt(0).toString();
			log.info("签名字符串：" + str);
			Map<String, Object> keyAndFactoryMap = RSAUtil.generateKeyAndFactory(publicKeyStr, true);
			Key key = RSAUtil.getKey(keyAndFactoryMap);
			PublicKey publicKey = (PublicKey) key;
			return RSAUtil.verifySign(str, sign, publicKey);
		}

		return false;
	}
}
